#! /usr/bin/env python1.5
#############################################################################
#
# Project:     GNUton
#
# File:        $Source: /home/arnold/CVS/gnuton/lib/GnutOS/Store.py,v $
# Version:     $RCSfile: Store.py,v $ $Revision: 1.2 $
# Copyright:   (C) 1998, David Arnold.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#############################################################################
"""
Store.py

User-visible Store implementation.  Instances of the Store class are
backed by an instance of StoreImpl.  StoreImpl is specialised for
different types of host device (like disc files, pcmcia cards, etc),
but Store remains the same.

Stores are created by the StorageManager when mounting a StoreImpl.

"""

#############################################################################

import random

from   GnutOS.Exceptions   import EvtExFrStore, HM_DeviceError
from   GnutOS.Platform     import kSError_WriteProtected, kFramesErrStoreIsROM
from   GnutOS.Primitive    import true, nil
from   GnutOS.Soup         import Soup
from   GnutOS.VBO          import VBO
from   GnutOS.Classes      import Frame


#############################################################################

class StoreException(HM_DeviceError):
    """Base class for all Store-related exceptions."""
    pass


#############################################################################

class StoreImpl:
    """

    Base class for Store implementations, where different
    implementations might be used for different device types (for
    example, a DiscStore might use a different implementation than a
    FlashStore).

    """

    def __init__(self):
	"""Initilise a Store object for the specified device."""
	pass

    def Format(self):
	"""Create a new Store instance, on the specified device."""
	pass

    def Erase(self):
	"""Erase the content of the Store."""
	pass

    def Open(self):
	"""Open an existing Store instance."""
	pass

    def Close(self):
	"""Close the Store, ready for unmounting."""
	pass

    def SetName(self, str_name):
	"""Set the name of this store."""
	pass

    def GetName(self):
	"""Return the name of this store."""
	pass

    def SetInfo(self, slotSymbol, value):
	pass

    def GetInfo(self, slotSymbol):
	pass

    def GetAllInfo(self):
	pass

    def GetSoup(self, soupNameString):
	pass

    def GetSoupNames(self):
	pass

    def HasSoup(self, soupName):
	pass

    def GetDevice(self):
	"""Return the FileDevice hosting this store."""
	pass

    def SetROM(self, flag):
	pass

    def SetWriteProtect(self, flag):
	pass

    def SetSignature(self, signature):
	pass

    def GetSignature(self):
	"""Return store signature."""
	pass

    def IsReadOnly(self):
	"""Return non-nil value if store cannot be written."""
	pass

    def CheckWriteProtect(self):
	"""Raise exception if store is locked or in ROM.

	Returns  --  unspecified

	If locked, raises *kSError_WriteProtected* (-10605) or if ROM,
	*kFramesErrStoreIsROM* (-48020). Note that current versions of
	GnutOS return *nil* is the store is writeable."""
	pass

    def TotalSize(self):
	"""Returns total size in bytes allowed for store on device."""
	pass

    def UsedSize(self):
	"""Returns the number of bytes used in the store."""
	pass




    def Soup(self):
	"""Return the Soup class for this type of Store."""
	return None

    def Index(self):
	"""Return the Index class for this type of Store."""
	return None

    def Entry(self):
	"""Return the Entry class for this type of Store."""
	return None


#############################################################################

class Store:
    """

    Visible Store class.  Store implementations are hidden by the
    StorageManager, and user programs refer to instances of this
    class.

    """

    def __init__(self, storageManager, storeImpl=None):
	"""Create a new user-visible Store object.

	*storageManager* -- reference to the Gnuton's StorageManager
	*storeImpl*      -- reference to the device-specific Store
	Returns          -- new, user-visible Store instance

	This method is called by the StorageManager when a formatted
	device is made available to the Gnuton (in the case of PCMCIA
	cards, this is when the card is inserted; in the case of
	disc-based stores, it's when the user tells the system to
	mount a particular store file).

	The HardwareManager determines that the device has been added
	to the active configuration, and that it contains a StoreImpl
	(device-specific store) instance.  It passes that instance to
	the StorageManager, where it is loaded and initialised.  Then
	an instance of this class is created referring to the
	StoreImpl instance, and notified to the user/system."""

	self._ref_sm = storageManager
	self._ref_impl = storeImpl
	return


    #-- soup methods

    def CreateSoupXmit(self, soupName, indexArray, changeSym):
	"""Create a soup and transmit a change notification.

	*soupName*    -- case-insensitive String up to 39 characters long.
	                 Must be unique among soups on the store.
	*indexArray*  -- Array of index specification frames, or nil.
	*changeSym*   -- Unique Symbol identifying the application that changed
	                 the soup, usually the application symbol.  Passing
			 nil for this parameter avoids notification.
	Returns       -- reference to new Soup object

	Creates a new soup."""

	if len(soupName) > 39:
	    #fixme: do this and determine error!
	    raise UnknownError("soup name too long, max 39 chars")

	if indexArray != nil and type(indexArray) != types.ListType:
	    raise UnknownError("bad type for indexArray parameter")

	#-- call implementation to create Soup implementation
	impl = self._ref_impl.CreateSoup(soupName)
	if indexArray != nil:
	    for f in indexArray:
		impl.AddIndex(f)

	#-- create abstract soup
	userSoup = Soup(self)

	#-- transmit notification
	if changeSym != nil:
	    self._ref_sm.XmitSoupChange(soupName, 
					changeSym,
					Symbol("soupCreated"),
					userSoup)
	return userSoup


    def Erase(self):
	"""Erase all soups on the Store."""

	self._ref_impl.Erase()
	return


    def GetSoup(self, soupNameString):
	"""Return the specified soup object from this store.

	*soupNameString* -- String name of soup
	Returns          -- soup object, or nil

	You can use this method to retrieve a union soup's members one
	at a time, but you cannot use this method to retrieve a union
	soup object; use the GetUnionSoupAlways() method for this
	purpose."""

	return self._ref_impl.GetSoup(soupNameString.python()) or nil


    def GetSoupNames(self):
	"""Return Array of Strings containing names of soups on the store."""

	names = Array()
	cnt = 0
	for name in self._ref_impl.GetSoupNames():
	    names[cnt] = String(name)
	    cnt = cnt + 1

	return names


    def HasSoup(self, soupName):
	"""Returns a non-nil value if the store contains the named soup.

	*soupName* -- String soup name
	Returns    -- true or nil

	While not specifed, a true result will return the true value.
	Code should *not* test for true, however: always test for
	non-nil."""

	return self._ref_impl.HasSoup(soupName.python()) and true or nil


    #--  Store status

    def IsReadOnly(self):
	"""Return non-nil value if store cannot be written.

	Returns  -- nil if store can be written, otherwise non-nil (true).

	Note that the Store could either be in ROM or be
	write-protected when read only."""

	return self._ref_impl.IsReadOnly and true or nil


    def CheckWriteProtect(self):
	"""Raise exception if store is locked or in ROM.

	Returns  --  unspecified

	If locked, raises *kSError_WriteProtected* (-10605) or if ROM,
	*kFramesErrStoreIsROM* (-48020). Note that current versions of
	GnutOS return *nil* is the store is writeable."""

	if not (self._wp or self._rom):
	    return nil

	elif not self._wp:
	    raise EvtExFrStore(kSError_WriteProtected)

	else:
	    raise ExtExFrStore(kFramesErrStoreIsROM)


    def IsValid(self):
	"""Returns true if the store can be used."""

	#fixme: before open/create and after close ?
	return true


    def SetName(self, storeNameString):
	self._name = storeStringName
	return


    def GetName(self):
	"""Return store name."""

	return self._name


    def GetSignature(self):
	"""Return store signature."""

	return self._sig


    def SetInfo(self, slotSymbol, value):
	"""Set the value of a slot in the Store's info frame."""

	self._info[slotSymbol] = value
	return nil


    def GetInfo(self, slotSymbol):
	"""Return the contents of the specified slot in the Store's info frame.

	*slotSymbol* -- string, slot name in info frame
	Returns      -- nil, or contents of named slot

	If the specified slot does not exist, this method returns
	*nil*."""

	if self._info.has_key(slotSymbol):
	    return self._info[slotSymbol]

	else:
	    return nil


    def GetAllInfo(self):
	"""Returns all contents of the Store info frame."""
	return copy.copy(self._info)



    def GetKind(self):
	"""Returns string description of storage type used by the store."""

	#fixme: this method needs to be a little more useful ...

	if self._internal:
	    return "Internal"

	else:
	    return "Storage card"


    def GetNextUid(self):
	return self._def["nextuid"]


    def TotalSize(self):
	"""Returns total size in bytes allowed for store on device."""

	return self._size


    def UsedSize(self):
	"""Returns the number of bytes used in the store."""

	return self._used


    #-- transactions

    def AtomicAction(self, myAction):
	"""Execute *myAction* as a transaction."""
	return


    def BusyAction(self, appSymbol, appName, myAction):
	"""Call *myAction* with the store marked busy."""
	return


    #--  packages methods

    def SuckPackageFromBinary(self, binary, paramFrame):
	""" """

	print "sucking package"

	cb = None
	freq = 0

	if paramFrame.has_key("callback"):
	    cb = paramFrame["callback"]

	if paramFrame.has_key("callbackFrequency"):
	    freq = paramFrame["callbackFrequency"]


	#-- load package! ;-)

	return


    def SuckPackageFromEndpoint(self, endPoint, paramFrame):
	return


    #-- VBO methods

    def NewVBO(self, klass, size):
	"""Create a new VBO in this store."""

	return VBO(self, klass, size)


    def NewCompressedVBO(self, klass, size, companderName, companderData):
	"""Create a new compressed VBO in this store."""

	return VBO(self, klass, size, companderName, companderData)


    #-- obsolete methods

    def CreateSoup(self, soupName, indexArray):
	""" """
	raise ObsoleteMethod("use CreateSoupFromSoupDef()")

    def RestorePackage(self, packageObject):
	"""Installs the specified package onto the store."""
	raise ObsoleteMethod("use SuckPackageFromBinary().")


#############################################################################

if __name__ == "__main__":
    pass


#############################################################################
